/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: thread.c,v 1.8.2.7 2003/10/29 11:39:16 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/thread.h>
#include <zorp/log.h>
#include <string.h>

gboolean use_threadpools = FALSE;
gint max_threads = 100;
gint num_threads = 0;
gint idle_threads = -1;
GPrivate *current_thread;

typedef struct _ZThreadCallback
{
  struct _ZThreadCallback *next;
  GFunc cb;
  gpointer user_data;
} ZThreadCallback;

static ZThreadCallback *start_callbacks;
static ZThreadCallback *stop_callbacks;

/**
 * z_thread_free:
 * @self: Zorp thread state
 *
 * Frees a given Zorp thread state.
 **/
static void
z_thread_free(ZThread *self)
{
  g_free(self);
}

/**
 * z_thread_self:
 * 
 * Returns the current Zorp specific thread state.
 **/
ZThread *
z_thread_self(void)
{
  return (ZThread *) g_private_get(current_thread);
}

/**
 * z_thread_register_start_callback:
 * @func: callback to add to the set of start-callbacks
 * @user_data: pointer to pass to this function
 *
 * Use this function to register a start-callback. Stop callbacks are called
 * when a given thread starts.
 **/
void
z_thread_register_start_callback(GFunc func, gpointer user_data)
{
  ZThreadCallback *cb = g_new0(ZThreadCallback, 1);

  cb->cb = func;
  cb->user_data = user_data;
  cb->next = start_callbacks;
  start_callbacks = cb;
}

/**
 * z_thread_register_stop_callback:
 * @func: callback to add to the set of stop-callbacks
 * @user_data: pointer to pass to this function
 *
 * Use this function to register a stop-callback. Stop callbacks are called
 * when a given thread terminates.
 **/
void
z_thread_register_stop_callback(GFunc func, gpointer user_data)
{
  ZThreadCallback *cb = g_new0(ZThreadCallback, 1);

  cb->cb = func;
  cb->user_data = user_data;
  cb->next = stop_callbacks;
  stop_callbacks = cb;
}


/**
 * z_thread_iterate_callbacks:
 * @self: thread state
 * @p: linked list to iterate through
 *
 * this function iterates on a linked list of registered callbacks and calls
 * each function.
 **/
static void
z_thread_iterate_callbacks(ZThread *self, ZThreadCallback *p)
{
  for (; p; p = p->next)
    p->cb(self, p->user_data);
}

/**
 * z_thread_func_core:
 * @self: thread specific variables
 * @user_data: pointer to pass to real thread function
 *
 * This function is called upon thread startup and performs thread specific
 * initialization. It calls thread-start and thread-exit callbacks.
 **/
static void
z_thread_func_core(ZThread *self, gpointer user_data)
{
  g_private_set(current_thread, self);
  self->thread = g_thread_self();

  z_thread_iterate_callbacks(self, start_callbacks);

  /*LOG
    This message indicates that a thread is starting.
  */
  z_log(self->name, CORE_DEBUG, 6, "thread starting;");
  (*self->func)(self->arg);
  /*LOG
    This message indicates that a thread is ending.
  */
  z_log(self->name, CORE_DEBUG, 6, "thread exiting;");

  z_thread_iterate_callbacks(self, stop_callbacks);  
  z_thread_free(self);

}

/* GLib threadpool based implementation */

static GThreadPool *pool;

/**
 * z_tp_thread_new:
 * @name: name to identify this thread with
 * @func: thread function
 * @arg:  pointer to pass to thread function
 *
 * Allocate and initialize a Zorp thread identified by a name, and
 * using the given thread function.
 *
 * Returns: TRUE to indicate success
 **/
gboolean
z_tp_thread_new(gchar *name, GThreadFunc func, gpointer arg)
{
  ZThread *self = g_new0(ZThread, 1);
  static gint thread_id = 1;
  GError *error = NULL;
  
  self->thread_id = thread_id++;
  self->func = func;
  self->arg = arg;
  strncpy(self->name, name, sizeof(self->name) - 1);

  g_thread_pool_push(pool, self, &error);
  
  if (error != NULL)
    z_log(NULL, CORE_ERROR, 4, "Some error occured when try to create a new thread; error='%s'", error->message);
  
  g_clear_error(&error);
  
  return TRUE;
}

/**
 * z_tp_thread_init: 
 *
 * initialize the threading subsystem
 **/
static void
z_tp_thread_init(void)
{
  g_thread_init(NULL);
  pool = g_thread_pool_new((GFunc) z_thread_func_core, NULL, max_threads, 0, NULL);
  if (idle_threads == -1)
    idle_threads = max_threads / 10;
  g_thread_pool_set_max_unused_threads(idle_threads);
}

/**
 * z_tp_thread_destroy:
 *
 * deinitialize the threading subsystem
 **/
static void
z_tp_thread_destroy(void)
{
  g_thread_pool_free(pool, TRUE, FALSE);
  pool = NULL;
}

/* Zorp threads implementation without using GLib's threadpools */

/* simple PThread based implementation */

static GAsyncQueue *queue;

/**
 * z_pt_thread_func:
 * @st: thread state
 *
 * This function wrapped around the real thread function logging two
 * events when the thread starts & stops 
 **/
static gpointer
z_pt_thread_func(gpointer st)
{
  ZThread *self = (ZThread *) st;
  
  do
    {
      z_thread_func_core(self, NULL);
      self = NULL;
      g_async_queue_lock(queue);
      self = (ZThread *) g_async_queue_try_pop_unlocked(queue);
      if (!self)
        {
          num_threads--;
          g_async_queue_unref_and_unlock(queue);
        }
      else
        g_async_queue_unlock(queue);
    }
  while (self != NULL);

  return NULL;
}

/**
 * z_pt_thread_new:
 * @name: name to identify this thread with
 * @func: thread function
 * @arg:  pointer to pass to thread function
 *
 * Allocate and initialize a Zorp thread identified by a name, and
 * using the given thread function.
 *
 * Returns: TRUE to indicate success
 **/
gboolean
z_pt_thread_new(gchar *name, GThreadFunc func, gpointer arg)
{
  ZThread *self = g_new0(ZThread, 1);
  GError *error = NULL;
  static gint thread_id = 1;
  
  self->thread_id = thread_id++;
  self->func = func;
  self->arg = arg;
  strncpy(self->name, name, sizeof(self->name) - 1);
  
  g_async_queue_lock(queue);
  if (num_threads >= max_threads)
    {
      /*LOG
        This message is logged when Zorp reaches the
        thread limit.
       */
      z_log(NULL, CORE_ERROR, 3, "Too many running threads, waiting for one to become free; num_threads='%d', max_threads='%d'", num_threads, max_threads);
      g_async_queue_push_unlocked(queue, self);
      g_async_queue_unlock(queue);
    }
  else
    {
      num_threads++;
      g_async_queue_ref_unlocked(queue);
      g_async_queue_unlock(queue);
      if (!g_thread_create(z_pt_thread_func, self, FALSE, &error))
        {
          z_log(NULL, CORE_ERROR, 2, "Error starting new thread; error='%s'", error->message);
          g_async_queue_lock(queue);
          num_threads--;
          g_async_queue_unlock(queue);
          return FALSE;
        }
    }
  
  return TRUE;
}

/**
 * z_pt_thread_init: 
 *
 * initialize the threading subsystem
 **/
void
z_pt_thread_init(void)
{
  g_thread_init(NULL);
  queue = g_async_queue_new();
  current_thread = g_private_new(NULL);
}

/**
 * z_pt_thread_destroy:
 *
 * deinitialize the threading subsystem
 **/
void
z_pt_thread_destroy(void)
{
  g_async_queue_unref(queue);
}

/* common interface */

/* FIXME: maybe we should use function pointers here instead */

gboolean
z_thread_new(gchar *name, GThreadFunc func, gpointer arg)
{
  if (use_threadpools)
    return z_tp_thread_new(name, func, arg);
  else
    return z_pt_thread_new(name, func, arg);
}

void
z_thread_init(void)
{
  if (use_threadpools)
    z_tp_thread_init();
  else
    z_pt_thread_init();

}

void
z_thread_destroy(void)
{
  if (use_threadpools)
    z_tp_thread_destroy();
  else
    z_pt_thread_destroy();
}

